#!/usr/bin/env python3

"""
 *   BSD LICENSE
 *
 *   Copyright (c) 2021 Samsung Electronics Co., Ltd.
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in
 *       the documentation and/or other materials provided with the
 *       distribution.
 *     * Neither the name of Samsung Electronics Co., Ltd. nor the names of
 *       its contributors may be used to endorse or promote products derived
 *       from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""

from rpc.client import print_dict, print_json, JSONRPCException
from rpc.helpers import deprecated_aliases

import logging
import argparse
import rpc
import sys

try:
    from shlex import quote
except ImportError:
    from pipes import quote


def print_array(a):
    print(" ".join((quote(v) for v in a)))


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description='DSS RPC command line interface')
    parser.add_argument('-s', dest='server_addr',
                        help='RPC server address', default='/var/tmp/spdk.sock')
    parser.add_argument('-p', dest='port',
                        help='RPC port number (if server_addr is IP address)',
                        default=5260, type=int)
    parser.add_argument('-t', dest='timeout',
                        help='Timout as a floating point number expressed in seconds waiting for reponse. Default: 60.0',
                        default=60.0, type=float)
    parser.add_argument('-r', dest='conn_retries',
                        help='Retry connecting to the RPC server N times with 0.2s interval. Default: 0',
                        default=0, type=int)
    parser.add_argument('-v', dest='verbose', action='store_const', const="INFO",
                        help='Set verbose mode to INFO', default="ERROR")
    parser.add_argument('--verbose', dest='verbose', choices=['DEBUG', 'INFO', 'ERROR'],
                        help="""Set verbose level. """)
    subparsers = parser.add_subparsers(help='RPC methods')

    def get_qpair_latency_profile(args):
        print_dict(rpc.latency_profile.get_qpair_latency_profile(args.client,
                                                      nqn=args.nqn,
                                                      cid=args.cid,
                                                      qid=args.qid))

    p = subparsers.add_parser('get_qpair_latency_profile', help='Get the latency profile infrmation of specific qpair')
    p.add_argument('-n', '--nqn', help='NVMe-oF target nqn')
    p.add_argument('-c', '--cid', help='NVMe-oF target subnqn controller ID', type=int)
    p.add_argument('-q', '--qid', help='NVMe-oF target subnqn controller qpair ID', type=int)
    p.set_defaults(func=get_qpair_latency_profile)

    def reset_ustat_counters(args):
        print_dict(rpc.latency_profile.reset_ustat_counters(args.client,
                                                      nqn=args.nqn))

    p = subparsers.add_parser('reset_ustat_counters', help='Reset ustat counters for NVMEoF subsystem')
    p.add_argument('-n', '--nqn', help='NVMe-oF target nqn')
    p.set_defaults(func=reset_ustat_counters)

    def dss_set_io_stall_timeout(args):
        print_dict(rpc.latency_profile.dss_set_io_stall_timeout(args.client,
                                                      timeout=args.timeout))
    p = subparsers.add_parser('dss_set_io_stall_timeout', help='Set stall IO timeout value if enabled')
    p.add_argument('-t', '--timeout', help='timeout value in seconds', type=int)
    p.set_defaults(func=dss_set_io_stall_timeout)

    def dss_rdb_compact(args):
        print_dict(rpc.latency_profile.dss_rdb_compact(args.client,
                                                      nqn=args.nqn,
                                                      get_status=args.get_status))

    p = subparsers.add_parser('rdb_compact', help='Compact rdb block devices for NVMEoF subsystem')
    p.add_argument('-n', '--nqn', help='NVMe-oF target nqn')
    p.add_argument('-s', '--get_status', help="Get only compaction status", action='store_true')
    p.set_defaults(func=dss_rdb_compact, get_status=False)

    args = parser.parse_args()


    try:
        args.client = rpc.client.JSONRPCClient(args.server_addr, args.port, args.timeout,
                                               log_level=getattr(logging, args.verbose.upper()),
                                               conn_retries=args.conn_retries)
        args.func(args)
    except JSONRPCException as ex:
        print("Exception:")
        print(ex.message)
        exit(1)
